<?php
/**
 * 星聚合登录 PHP SDK
 * 
 * @author 星聚合登录
 * @version 1.0.0
 */

class OAuth
{
    private $apiurl;
    private $appid;
    private $appkey;
    private $callback;

    /**
     * 构造函数
     * 
     * @param array $config 配置参数
     *   - apiurl: API地址
     *   - appid: 应用ID
     *   - appkey: 应用密钥
     *   - callback: 回调地址
     */
    public function __construct(array $config)
    {
        $this->apiurl = rtrim($config['apiurl'] ?? '', '/');
        $this->appid = $config['appid'] ?? '';
        $this->appkey = $config['appkey'] ?? '';
        $this->callback = $config['callback'] ?? '';
    }

    /**
     * 发起登录请求
     * 
     * @param string $type 登录类型 (qq/wx/alipay/sina/github等)
     * @param string $state 自定义参数，回调时原样返回
     * @return array
     */
    public function login($type, $state = '')
    {
        $params = [
            'act' => 'login',
            'appid' => $this->appid,
            'appkey' => $this->appkey,
            'type' => $type,
            'redirect_uri' => $this->callback,
            'state' => $state,
        ];

        return $this->request($params);
    }

    /**
     * 获取用户信息
     * 
     * @param string $code 授权码
     * @return array
     */
    public function callback($code)
    {
        $params = [
            'act' => 'callback',
            'appid' => $this->appid,
            'appkey' => $this->appkey,
            'code' => $code,
        ];

        return $this->request($params);
    }

    /**
     * 查询用户信息
     * 
     * @param string $type 登录类型
     * @param string $socialUid 用户唯一标识
     * @return array
     */
    public function query($type, $socialUid)
    {
        $params = [
            'act' => 'query',
            'appid' => $this->appid,
            'appkey' => $this->appkey,
            'type' => $type,
            'social_uid' => $socialUid,
        ];

        return $this->request($params);
    }

    /**
     * 发送请求
     * 
     * @param array $params 请求参数
     * @return array
     */
    private function request(array $params)
    {
        $url = $this->apiurl . '/connect.php?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['code' => -1, 'msg' => 'CURL Error: ' . $error];
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['code' => -1, 'msg' => 'JSON Parse Error'];
        }
        
        return $result;
    }
}
