"""
星聚合登录 Python SDK

@author 星聚合登录
@version 1.0.0
"""

import requests
from urllib.parse import urlencode


class OAuth:
    """OAuth 聚合登录客户端"""

    def __init__(self, apiurl: str, appid: str, appkey: str, callback: str):
        """
        初始化OAuth客户端

        Args:
            apiurl: API地址
            appid: 应用ID
            appkey: 应用密钥
            callback: 回调地址
        """
        self.apiurl = apiurl.rstrip('/')
        self.appid = appid
        self.appkey = appkey
        self.callback = callback

    def login(self, type: str, state: str = '') -> dict:
        """
        发起登录请求

        Args:
            type: 登录类型 (qq/wx/alipay/sina/github等)
            state: 自定义参数，回调时原样返回

        Returns:
            dict: API返回结果
        """
        params = {
            'act': 'login',
            'appid': self.appid,
            'appkey': self.appkey,
            'type': type,
            'redirect_uri': self.callback,
            'state': state,
        }
        return self._request(params)

    def callback(self, code: str) -> dict:
        """
        获取用户信息

        Args:
            code: 授权码

        Returns:
            dict: 用户信息
        """
        params = {
            'act': 'callback',
            'appid': self.appid,
            'appkey': self.appkey,
            'code': code,
        }
        return self._request(params)

    def query(self, type: str, social_uid: str) -> dict:
        """
        查询用户信息

        Args:
            type: 登录类型
            social_uid: 用户唯一标识

        Returns:
            dict: 用户信息
        """
        params = {
            'act': 'query',
            'appid': self.appid,
            'appkey': self.appkey,
            'type': type,
            'social_uid': social_uid,
        }
        return self._request(params)

    def _request(self, params: dict) -> dict:
        """
        发送请求

        Args:
            params: 请求参数

        Returns:
            dict: API返回结果
        """
        url = f"{self.apiurl}/connect.php?{urlencode(params)}"
        try:
            response = requests.get(url, timeout=30)
            return response.json()
        except Exception as e:
            return {'code': -1, 'msg': str(e)}
